//-----------------------------------------------------------------------------
//Copyright  2000-2003 TogetherSoft Corporation, a wholly owned subsidiary of Borland Software Corporation. All rights reserved. All patents pending.
//-----------------------------------------------------------------------------

using System;
using System.Collections;

namespace CSCashSales.data_objects
{
	/// <summary>
	///    This is the container of cash sales.
	///    It implements ICashSaleContainer and IPresentView interfaces.
	/// </summary>
	/// <TgData>
	/// </TgData>
    public class CashSaleContainer : ICashSaleContainer, IPresentView
    {
		  // Codes of table attributes for IPresentView.
      /// <summary>
      /// Code ATTR_DATE
      /// </summary>
		  public const int ATTR_DATE = 0;
      /// <summary>
      /// Code ATTR_TOTAL
      /// </summary>
		  public const int ATTR_TOTAL = 1;
      /// <summary>
      /// Code ATTR_SUBTOTAL
      /// </summary>
		  public const int ATTR_SUBTOTAL = 2;
      /// <summary>
      /// Code ATTR_TAX
      /// </summary>
		  public const int ATTR_TAX = 3;

		  private const int ATTR_COUNT = 4;

		  // Codes of linear attributes for IPresentView.
      /// <summary>
      /// Code LATTR_PAYMENT
      /// </summary>
		  public const int LATTR_TOTAL = 0;
      /// <summary>
      /// Code LATTR_PAYMENT
      /// </summary>
		  public const int LATTR_SUBTOTAL = 1;
      /// <summary>
      /// Code LATTR_PAYMENT
      /// </summary>
		  public const int LATTR_TAX = 2;

		  private const int LATTR_COUNT = 3;

		  /// <summary>
		  /// A container of cash sales.
		  /// </summary>
		  private ArrayList m_objList;

		  /// <summary>
		  /// Constructor.
		  /// </summary>
      public CashSaleContainer()
      {
			  m_objList = new ArrayList();
      }

		/// <summary>
		/// Returns a number of cash sales.
		/// </summary>
		public int getCount() 
		{
			return m_objList.Count;
		}

		/// <summary>
		/// An access to container items by its indexes.
		/// </summary>
		/// <param name="index"> The item index </param>
		public IDataItem getDataItemAt( int index ) 
		{
			return m_objList[index]	as IDataItem;
		}

		/// <summary>
		/// Adds the data item to the container.
		/// </summary>
		/// <param name="newItem"> The data item </param>
		public void addDataItem( IDataItem newItem ) 
		{
			m_objList.Add(newItem);
		}

		/// <summary>
		/// Generates new free ID.
		/// </summary>
        public int generateID() 
		{
			int result = 1;
			for(int i=0; i<getCount(); i++)
			{
				IDataItem item = getDataItemAt(i);
				if(item.getID() > result)
					result = item.getID();
			}
			result++;
			return result;	
		}

		/// <summary>
		/// An access to cash sales by its indexes.
		/// </summary>
		/// <param name="index"> The sale index </param>
		public ICashSale getCashSaleAt( int index ) 
		{
			return getDataItemAt(index) as ICashSale;
		}

		/// <summary>
		/// Adds new cash sale.
		/// </summary>
		/// <param name="newCashSale"> The cash sale </param>
		public void addCashSale( ICashSale newCashSale ) 
		{
			addDataItem(newCashSale);
		}

		/// <summary>
		/// Removes the cash sale.
		/// </summary>
		/// <param name="objCashSale"> The cash sale </param>
		public bool removeCashSale( ICashSale objCashSale ) 
		{
			int index = m_objList.IndexOf(objCashSale);
			if(index >= 0)
				m_objList.RemoveAt(index);
			return index >= 0;
		}

		/// <summary>
		/// Returns the subtotal amount.
		/// </summary>
		public Decimal getSubTotal() 
		{
			Decimal result = new Decimal(0);
			for(int i=0; i<getCount(); i++)
				result += getCashSaleAt(i).getSubTotal();
			return result;
		}

		/// <summary>
		/// Returns the tax amount.
		/// </summary>
		public Decimal getTax() 
		{
			Decimal result = new Decimal(0);
			for(int i=0; i<getCount(); i++)
				result += getCashSaleAt(i).getTax();
			return result;
		}

		/// <summary>
		/// Returns the total amount.
		/// </summary>
		public Decimal getTotal() 
		{
			Decimal result = new Decimal(0);
			for(int i=0; i<getCount(); i++)
				result += getCashSaleAt(i).getTotal();
			return result;
		}

		/// <summary>
		/// Indicates whether the object has linear presentable attributes.
		/// </summary>
		public bool hasLinearAttributes() 
		{
			return true;
		}

		/// <summary>
		/// Indicates whether the object has table presentable attributes.
		/// </summary>
		public bool hasTableAttributes() 
		{
			return true;
		}

		/// <summary>
		/// Returns the number of linear presentable attributes.
		/// </summary>
		public int getLinearAttrCount() 
		{
			return LATTR_COUNT;
		}

		/// <summary>
		/// Returns the name of a linear presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getLinearAttrName( int attr ) 
		{
			switch(attr)
			{
				case LATTR_TOTAL:
					return "Total";
				case LATTR_SUBTOTAL:
					return "Subtotal";
				case LATTR_TAX:
					return "Tax";
				default:
					return null;
			}
		}

		/// <summary>
		/// Returns the value of a linear presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getLinearAttrValue( int attr ) 
		{
			switch(attr)
			{
				case LATTR_TOTAL:
					//return Decimal.ToString(getTotal());
					return getTotal().ToString();
				case LATTR_SUBTOTAL:
					//return Decimal.ToString(getSubTotal());
					return getSubTotal().ToString();
				case LATTR_TAX:
					//return Decimal.ToString(getTax());
					return getTax().ToString();
				default:
					return null;
			}
		}

		/// <summary>
		/// Returns the number of table presentable attributes.
		/// </summary>
		public int getTableAttrCount() 
		{
			return ATTR_COUNT;
		}

		/// <summary>
		/// Returns the size of attribute table.
		/// </summary>
		public int getTableRowCount() 
		{
			return getCount();	
		}

		/// <summary>
		/// Returns the name of a table presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getTableAttrName( int attr ) 
		{
			switch(attr)
			{
				case ATTR_DATE:
					return "Date";
				case ATTR_TOTAL:
					return "Total";
				case ATTR_SUBTOTAL:
					return "Subtotal";
				case ATTR_TAX:
					return "Tax";
				default:
					return null;
			}
		}

		/// <summary>
		/// Returns the value of a table presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		/// <param name="row"> The table row </param>
		public String getTableAttrValue( int attr, int row ) 
		{
			ICashSale sale = getCashSaleAt(row);
			switch(attr)
			{
				case ATTR_DATE:
					return sale.getSaleDate().ToString();
				case ATTR_TOTAL:
					return sale.getTotal().ToString();
				case ATTR_SUBTOTAL:
					return sale.getSubTotal().ToString();
				case ATTR_TAX:
					return sale.getTax().ToString();
				default:
					return null;
			}
		}

    }
}
